// IdentityMatchExperimenterPlugin.cpp starts

#define		NUMBER_MAZE_ARMS				8

#define		NUMBER_FAMILIARIZATION_TRIALS	2

// Maximum number of arm selections when revisiting an arm (see documentation below).
#define		MAX_REVISIT_SELECTIONS			2

// Number of trials over which to evaluate the learning criterion (see documentation below).
#define		NUMBER_CRITERION_TRIALS			8

// Max number of arm choices allowed in familiarization stage
#define		MAX_FAM_NUMBER_ARM_CHOICES		NUMBER_MAZE_ARMS*3

// Maximum number of arm choices allowed before taking the rat off the maze.
#define		MAX_NUMBER_ARM_CHOICES			NUMBER_MAZE_ARMS*2

#include <Message.h>
#include <stdio.h>
#include "IdentityMatchExperimenterPlugin.h"
#include "EnvironmentConstants.h"
#include "Debug.h"
#include "UserEnvMessage.h"
#include "PluginSettingsNames.h"
#include "DatabaseFieldNames.h"
#include "PerfQueue.h"

/*******************************************************
* Created: 12/20/00, CGP
* 	Plugin for Spatial Identity Match to Sample experimental procedure.
*
*	Task Synopsis:
*   --------------
*		This task has the rat learning to re-visit a location. Testing is
*	also carried out to see how well the rat generalizes this learning
*	to arms of the maze that were not used in training.
*
*	Task:
*   -----
*		The rat is first given a familiarization stage in which the rat goes on the maze
*   for NUMBER_FAMILIARIZATION_TRIALS trials with all arms baited. In each of these trials
*	the rat is taken off the maze after consuming all of the food or making
*	at most MAX_FAM_NUMBER_ARM_CHOICES. After familiarization, the rat enters into
*   a training stage. In each training trial, the rat goes on the maze twice.
*   	When the rat goes on the maze the first time for a training trial,
*   arm X is baited with food (one of the arms on the maze).
*   The rat's task is now to find the arm that is baited with food
*   and eat the food. The second time the rat goes on the maze for the training
*	trial, arm X is re-baited, and the rat is supposed to learn that arm X is re-baited
*	so he should re-visit that same arm.
*   	Successive training trials are given where arm X is selected
*   from amongst the set of training arms (say, four selected arms of an
*   8-arm radial-maze).
*		The rat goes from the training stage of the procedure to the testing
*	stage of the procedure when he meets the learning criterion. The learning criterion
*	is computed by keeping track of the number of arm selections the rat takes to find the food
*   when he goes on the maze for the second time in a training trial. The rat is said to have
*	met the learning criterion if he takes on average at most MAX_REVISIT_SELECTIONS arm selections
*	across each of the last NUMBER_CRITERION_TRIALS trials.
*		After meeting the learning criterion in training, the rat proceeds to the
*	testing stage. In testing, the rat receives trials both on arms that were used in
*	training and also the collection of arms that were not used in training
*	(the "testing arms"). For example, in training with an 8-arm radial-maze
*	arm X might be selected from N, NE, W, SE, and in testing arm X would be
*	any one of the 8 arms on the maze.
*		Testing proceeds until the learning criterion is met in the last
*	NUMBER_CRITERION_TRIALS trials run on the testing arms.
*
*******************************************************/

ExperimenterPlugin *instantiate_exp(PortMessage *expRatPort,
      		PortMessage *envExpPort, DatabaseMessage *dbMsg, UserEnvMessage *userEnvMsg,
      		DebugServer *bugServer) {
   return new IdentityMatchExperimenterPlugin(expRatPort, envExpPort, dbMsg, userEnvMsg,
   		bugServer);
}

IdentityMatchExperimenterPlugin::IdentityMatchExperimenterPlugin(PortMessage *expRatPort,
      		PortMessage *envExpPort, DatabaseMessage *dbMsg, UserEnvMessage *userEnvMsg,
      		DebugServer *bugServer):
				ExperimenterPlugin(expRatPort, envExpPort, dbMsg, userEnvMsg, bugServer),
				perfQueue(NUMBER_CRITERION_TRIALS)
{
	// Temporary: will access settings from the database when we have a GUI dialog that enables
	// changing the plugin settings.
	settings = new BMessage();
	
	// create default plugin settings, if none given
	if (settings->CountNames(B_ANY_TYPE) == 0) {
		Debug('n', "IdentityMatchExperimenterPlugin::IdentityMatchExperimenterPlugin: Null default settings");
		this->settings->AddInt16(NUMBER_OF_MAZE_ARMS, NUMBER_MAZE_ARMS);
		EXP_SaveSettings(this->settings);
	} // default constructor already established value for this->settings otherwise

	// NOTE: We can't set up the maze here because we don't get access to the environment
	// in the constructor. Put the maze setup in the "Start" function.
}

IdentityMatchExperimenterPlugin::~IdentityMatchExperimenterPlugin()
{
}

// Set up the maze for the rat plugin
void IdentityMatchExperimenterPlugin::SetupForTrial()
{
	int i;
	
	// Open all the doors.
	for (i=0; i < NUMBER_MAZE_ARMS; i++) {
		EXP_OpenDoor(i);
		EXP_RemoveFood(i);
	}

	EXP_PlaceRatAt(CENTER);
}

// Training data sequence. Each number in this sequence gives an arm to be used at a particular
// trial in training.
// Training arms: 0, 1, 3, 6
static int train[] = { 0, 1, 3, 6 }; 

static int test[] = { 2, 4, 5, 7 };

int GetMaxTrainTrial()
{
	return sizeof(train)/sizeof(int);
}

int GetMaxTestTrial()
{
	return sizeof(test)/sizeof(int);
}
	
// Get the arm to be used for a particular trial number. That is, find out the
// arm that will be used for the trial. TrainOrTest is "Train"
// for training, and "Test" for testing. trialNumber starts at 0 and goes up to GetMaxTrainTrial()
// or GetMaxTestTrial(). Returns -1 if there was an error.
int IdentityMatchExperimenterPlugin::GetArm(char *mode, int trialNumber)
{
	int arm = -1;
	
	if (strcmp(mode, "Train") == 0) {
		if (trialNumber > GetMaxTrainTrial()) {
			Debug('e', "IdentityMatchExperimenterPlugin::GetArm: Max train trial exceeded");
			return arm;
		}
		arm = train[trialNumber];
	} else if (strcmp(mode, "Test") == 0) {
		if (trialNumber > GetMaxTestTrial()) {
			Debug('e', "IdentityMatchExperimenterPlugin::GetArm: Max test trial exceeded");
			return arm;
		}
		arm = test[trialNumber];
	} else {
		Debug('e', "IdentityMatchExperimenterPlugin::GetArm: Bad flag: ", mode);
	}
	
	return arm;
}

/* Function to run a single trial for the rat. Assumes that a single arm of the maze (goalArm)
   has been baited for the trial, and that maze arms are open.
   Returns the number of arm choices (>= 1) that the rat took to find the food.
   The rat is taken off the maze if he makes MAX_NUMBER_ARM_CHOICES arm choices.
*/
int IdentityMatchExperimenterPlugin::RunTrialProtocol(int goalArm, int trialPhase)
{
	int numChoices = 0;
	bool foodFound = false;
	
	for (numChoices=1; numChoices < MAX_NUMBER_ARM_CHOICES; numChoices++) {
		Debug('n', "IdentityMatchExperimenterPlugin::RunTrialProtocol: Rat made arm choice");

		// Wait until rat enters an arm: May be goalArm or some other arm
		int actualChoiceArm = EXP_WaitForEvent(EXP_EVENT_ARM_ENTRY);
		numChoices++;
		
		// Rat will next either eat the food down the arm because he made a correct arm choice
		// or will enter the center of the maze. Wait for this event.
		int eventVal = EXP_WaitForEvent(EXP_EVENT_CENTER_ENTRY | EXP_EVENT_FOOD_CONSUMED);
	
		Debug('n', "IdentityMatchExperimenterPlugin::RunTrialProtocol: Rat entered center or ate food");

		bool foundFood = false;
		bool consumedFood = false;
		
		if (eventVal != CENTER) { // rat did not enter center, therefore, he ate the food
			// Rat entered correct arm and ate food
			foundFood = true;
			consumedFood = true;
		} else { // rat entered center
			if (actualChoiceArm == goalArm) {
				// Rat entered correct arm but did not eat food
				foodFound = true;
			}
		}
		
		AddPerformanceRecord(trialPhase, goalArm, actualChoiceArm, foundFood, consumedFood);
		
		if (foundFood) break;
	}

	return numChoices;
}

// armNumber == actual arm chosen by rat
void IdentityMatchExperimenterPlugin::AddPerformanceRecord(int trialPhase, int goalArm,
	int armNumber, int foundFood, int consumedFood)
{
	BMessage *perfDataRecord = new BMessage();
	
	// Database only allows int16's right now as data type
	perfDataRecord->AddInt16(DBFLD_RECORD_TYPE, 4);
	perfDataRecord->AddInt16(DBFLD_RECORD_SUB_TYPE, 1);
	perfDataRecord->AddInt16(DBFLD_TRIAL_PHASE, (int16) trialPhase);
	perfDataRecord->AddInt16(DBFLD_ARM_NUMBER, goalArm);
	perfDataRecord->AddInt16(DBFLD_RAT_NUMBER, ratNo);
	perfDataRecord->AddInt16(DBFLD_TRIAL_NUMBER, trialNumber);
	perfDataRecord->AddInt16(DBFLD_ARM_NUMBER, armNumber);
	perfDataRecord->AddInt16(DBFLD_CONTAINED_FOOD, foundFood);
	perfDataRecord->AddInt16(DBFLD_CONSUMED_FOOD, consumedFood);
	EXP_AddDatabaseRecord(perfDataRecord);
	delete perfDataRecord;
	
	Debug('n', "IdentityMatchExperimenterPlugin::AddPerformanceRecord: Saved record");
}

bool IdentityMatchExperimenterPlugin::TrainCriterionSatisified()
{
	int numTrials = perfQueue.CurrLength();
	
	if (numTrials < NUMBER_CRITERION_TRIALS) {
		// can only satisfy criterion if we've run a certain number of trials
		return false;
	}

	// performance in the queue is total number of arm selections across trials
	return (perfQueue.Performance() <= MAX_REVISIT_SELECTIONS*NUMBER_CRITERION_TRIALS);
}

bool IdentityMatchExperimenterPlugin::TestCriterionSatisified()
{
	return TrainCriterionSatisified();
}

void IdentityMatchExperimenterPlugin::DoNextFamiliarizationTrial()
{
	int numFoodEaten = 0;
	int numChoices = 0;
	
	// Bait maze
	for (int i=0; i < NUMBER_MAZE_ARMS; i++) {
		EXP_PlaceFood(i);
	}

	EXP_PlaceRatAt(CENTER);
	
	// Put rat on maze and let him run around
	EXP_PutRatOnMaze();
	
	// Take him off the maze when he makes MAX_FAM_NUMBER_ARM_CHOICES or after consuming
	// all the food.
	for (;;) {
		int event = EXP_WaitForEvent(EXP_EVENT_FOOD_CONSUMED | EXP_EVENT_RAT_MOVED);
		
		if (event >= 0) { // valid arm #, must have entered an arm
			numChoices++;
		} else if (event != CENTER) { // invalid arm number & not center entry, must have eaten food
			numFoodEaten++;
		}
		
		if (numChoices >= MAX_FAM_NUMBER_ARM_CHOICES) break;
		if (numFoodEaten == NUMBER_MAZE_ARMS) break;
	}
	
	EXP_TakeRatOffMaze(24.0); // 24 hrs between familiarization trials
	EXP_PlaceRatAt(CENTER); // just for aesthetics of GUI
	
	trialNumber++;
	if (trialNumber > NUMBER_FAMILIARIZATION_TRIALS) {
		mode = "Train";
		trialNumber = 1;
	}
}

// Run next familiarization, training or testing trial. Returns true iff we've finished all
// familiarization, training and testing for the current rat.
bool IdentityMatchExperimenterPlugin::RunNextTrial()
{
	int goalArm;
	bool doneRat = false;
	int numChoices;
	
	SetupForTrial();
	
	if (strcmp(mode, "Familiarization") == 0) {
		// Setup and run next familiarization trial
		DoNextFamiliarizationTrial();
		return false;
	}

	// Get the goal arm to be used on this trial.
	goalArm = GetArm(mode, trialNumber);
	
	////////////////////////////////////////////////////
	//////// Do the first part of the trial ////////////
	////////////////////////////////////////////////////

	EXP_PlaceFood(goalArm);
	EXP_PutRatOnMaze();
	
	(void) RunTrialProtocol(goalArm, 1); // phase 1
	
	EXP_TakeRatOffMaze(0.1); // take rat off maze briefly to rebait
	EXP_PlaceRatAt(CENTER);
	
	/////////////////////////////////////////////////////
	//////// Do the second part of the trial ////////////
	/////////////////////////////////////////////////////

	EXP_PlaceFood(goalArm);
	EXP_PutRatOnMaze();
	
	numChoices = RunTrialProtocol(goalArm, 2); // phase 2
	
	perfQueue.Enqueue(numChoices); // track performance: number of arm selections per trial
	EXP_TakeRatOffMaze(24.0); // take rat off maze for 24.0 hours
	EXP_PlaceRatAt(CENTER); // for aesthetics between trials
	
	trialNumber++;
	
	if (strcmp(mode, "Train") == 0) {
		// Switch from training to testing?
		if ((trialNumber > GetMaxTrainTrial()) || TrainCriterionSatisified()) {
			mode = "Test";
			trialNumber = 1;
			perfQueue.Reset(); // clear training data from performance queue
			doneRat = true; // turn off testing for now
		}
	} else if (strcmp(mode, "Test") == 0) {
		// Finished testing?
		if ((trialNumber > GetMaxTestTrial()) || TestCriterionSatisified()) {
			doneRat = true;
		}
	} else {
		Debug('e', "IdentityMatchExperimenterPlugin::RunNextTrial: Error: Bad mode= ", mode);
	 	doneRat = true;
	}

	return doneRat;
}

// Setup the ratNo and trialNumber for the first rat run.
void IdentityMatchExperimenterPlugin::FirstRat()
{	
	// Need to get the current rat number from the database.
	ratNo = EXP_GetCurrentRatNumber();
	if (ratNo <= 0) {
		ratNo = 1;
		trialNumber = 1;
	} else {
		NextRat();
	}
}

// Increments the current rat number. Return true iff there is a next rat to run.
bool IdentityMatchExperimenterPlugin::NextRat()
{
	// Check to see if we've done all the trials
	if (ratNo < EXP_GetNumberOfRats()) {
		ratNo++;
		trialNumber = 1;
		EXP_SetCurrentRatNumber(ratNo);
		return true;
	} else {
		EXP_UserMessage("No more rats to run.");
		return false;
	}
}

void IdentityMatchExperimenterPlugin::PLUGIN_Setup()
{
	Debug('n', "IdentityMatchExperimenterPlugin::PLUGIN_Setup");
	
	ratNo = 0;
	FirstRat();
	mode = "Familiarization";
}

void IdentityMatchExperimenterPlugin::PLUGIN_RunTrial()
{
	Debug('n', "IdentityMatchExperimenterPlugin::PLUGIN_RunTrial");
	
	if (RunNextTrial()) {
		NextRat();
	}
}

void IdentityMatchExperimenterPlugin::PLUGIN_RunCurrentRat()
{
	// Complete trials for current rat
	Debug('n', "IdentityMatchExperimenterPlugin::PLUGIN_RunCurrentRat");	
	while (! RunNextTrial()) {
	}
	NextRat();
}

void IdentityMatchExperimenterPlugin::PLUGIN_RatConsumedFood()
{
	Debug('n', "IdentityMatchExperimenterPlugin::PLUGIN_RatConsumedFood");
}

void IdentityMatchExperimenterPlugin::PLUGIN_RatMoved(BMessage *msg)
{
	Debug('n', "IdentityMatchExperimenterPlugin::PLUGIN_RatMoved");
}

// IdentityMatchExperimenterPlugin.cpp ends

